<?php

// Aumenta il tempo di esecuzione massimo e il limite di memoria
ini_set('max_execution_time', 300); // 300 seconds = 5 minutes
ini_set('memory_limit', '512M'); // Increase memory limit

function logMessage($message) {
    $logFile = __DIR__ . '/debug.log';
    $time = date('Y-m-d H:i:s');
    file_put_contents($logFile, "[$time] $message\n", FILE_APPEND);
}

// Database connection settings
$host = 'localhost';
$dbname = 'enfasi_farmajoin';
$username = 'enfasi_farmajoin';
$password = 'farmajoin105';

try {
    $pdo = new PDO("mysql:host=$host;dbname=$dbname", $username, $password);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    logMessage("Connessione al database riuscita.");
} catch (PDOException $e) {
    logMessage("Errore di connessione al database: " . $e->getMessage());
    exit;
}

$output_dir = __DIR__ . '/scaricati/';
$output_file = $output_dir . 'joinedData.xml';
$tables = [
    'TE002' => 'FDI_0001',
    'TE004' => 'FDI_T456',
    'TE018' => 'FDI_0001',
    'TR039' => 'FDI_0001',
    'TR048' => 'FDI_0001',
    'TR011' => 'FDI_0001',
    'TR036' => 'FDI_T045',
    'TR037' => 'FDI_0001',
    'TS010' => 'FDI_T001',
    'TS011' => 'FDI_T004',
    'TS042' => 'FDI_T096',
    'TS043' => 'FDI_T047',
    'TS054' => 'FDI_T508',
    'TS077' => 'FDI_T618'
];

// Delete previous output file if it exists
if (file_exists($output_file)) {
    unlink($output_file);
    logMessage("File precedente eliminato: $output_file");
} else {
    logMessage("Nessun file precedente trovato: $output_file");
}

// Drop existing tables in the database
foreach (array_keys($tables) as $table) {
    $pdo->exec("DROP TABLE IF EXISTS $table");
    logMessage("Tabella $table eliminata se esistente.");
}

// Function to create a table based on XML structure
function createTableFromXML($pdo, $table, $filePath) {
    $xml = simplexml_load_file($filePath);
    if (!$xml) {
        logMessage("Errore nel caricamento del file XML: $filePath");
        exit;
    }

    $columns = [];
    foreach ($xml->RECORD[0] as $key => $value) {
        $columns[] = "$key VARCHAR(255)";
    }
    $columns_sql = implode(", ", $columns);

    $sql = "CREATE TABLE $table (
        id INT AUTO_INCREMENT PRIMARY KEY,
        $columns_sql
    )";
    $pdo->exec($sql);
    logMessage("Tabella $table creata con le colonne: $columns_sql.");
}

// Function to insert data from XML to the table in batches
function insertDataFromXML($pdo, $table, $filePath, $batchSize = 100) {
    $xml = simplexml_load_file($filePath);
    if (!$xml) {
        logMessage("Errore nel caricamento del file XML: $filePath");
        exit;
    }

    $columns = [];
    foreach ($xml->RECORD[0] as $key => $value) {
        $columns[] = $key;
    }
    $columns_sql = implode(", ", $columns);

    $batch = [];
    $count = 0;

    foreach ($xml->RECORD as $record) {
        $value_set = [];
        foreach ($columns as $column) {
            $value_set[] = $pdo->quote((string)$record->$column);
        }
        $batch[] = "(" . implode(", ", $value_set) . ")";
        $count++;

        if ($count % $batchSize == 0) {
            $sql = "INSERT INTO $table ($columns_sql) VALUES " . implode(", ", $batch);
            $pdo->exec($sql);
            $batch = [];
            logMessage("Inserito batch di $batchSize record nella tabella $table.");
        }
    }

    if (!empty($batch)) {
        $sql = "INSERT INTO $table ($columns_sql) VALUES " . implode(", ", $batch);
        $pdo->exec($sql);
        logMessage("Inseriti " . count($batch) . " record rimanenti nella tabella $table.");
    }

    // Verifica il numero di record inseriti
    $stmt = $pdo->query("SELECT COUNT(*) AS count FROM $table");
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    logMessage("Numero di righe nella tabella $table: " . $row['count']);

    if ($row['count'] == 0) {
        logMessage("Nessun dato inserito nella tabella $table");
        exit;
    }
}

// Creazione delle tabelle e importazione dei dati
foreach ($tables as $table => $keyField) {
    createTableFromXML($pdo, $table, $output_dir . $table . '.xml');
    insertDataFromXML($pdo, $table, $output_dir . $table . '.xml');
}

logMessage("Tutti i dati sono stati caricati nel database.");

// Creazione della tabella dei risultati uniti
$joinedTable = 'joined_data';
$pdo->exec("DROP TABLE IF EXISTS $joinedTable");

$createJoinedTableSQL = "CREATE TABLE $joinedTable (
    id INT AUTO_INCREMENT PRIMARY KEY,
    keyValue VARCHAR(255) NOT NULL,
    " . implode(", ", array_map(function ($table) use ($pdo, $tables) {
    $columns = $pdo->query("DESCRIBE $table")->fetchAll(PDO::FETCH_COLUMN);
    unset($columns[0]); // Remove 'id' column
    return implode(", ", array_map(function ($column) use ($table) {
        return "$table" . "_" . "$column VARCHAR(255)";
    }, $columns));
}, array_keys($tables))) . "
)";
$pdo->exec($createJoinedTableSQL);
logMessage("Tabella joined_data creata.");

// Esecuzione dei join necessari nel database e inserimento dei risultati uniti per i primi 100 record in batch di 20
$batchSize = 20;
$offset = 0;
$totalRows = $pdo->query("SELECT COUNT(*) FROM TE002")->fetchColumn();
logMessage("Numero totale di record in TE002: $totalRows");

$start = microtime(true);

while ($offset < 100 && $offset < $totalRows) {
    $selectColumns = implode(", ", array_map(function ($table) use ($pdo, $tables) {
        $columns = $pdo->query("DESCRIBE $table")->fetchAll(PDO::FETCH_COLUMN);
        unset($columns[0]); // Remove 'id' column
        return implode(", ", array_map(function ($column) use ($table) {
            return "$table.$column AS " . "$table" . "_" . "$column";
        }, $columns));
    }, array_keys($tables)));

    $joinSQL = "INSERT INTO $joinedTable (keyValue, $selectColumns)
                SELECT TE002.FDI_0001 AS keyValue, $selectColumns
                FROM TE002
                LEFT JOIN TE004 ON TE002.FDI_0001 = TE004.FDI_T456
                LEFT JOIN TE018 ON TE002.FDI_0001 = TE018.FDI_0001
                LEFT JOIN TR039 ON TE002.FDI_0001 = TR039.FDI_0001
                LEFT JOIN TR048 ON TE002.FDI_0001 = TR048.FDI_0001
                LEFT JOIN TR011 ON TE002.FDI_0001 = TR011.FDI_0001
                LEFT JOIN TR036 ON TE002.FDI_0001 = TR036.FDI_T045
                LEFT JOIN TR037 ON TE002.FDI_0001 = TR037.FDI_0001
                LEFT JOIN TS010 ON TE002.FDI_0001 = TS010.FDI_T001
                LEFT JOIN TS011 ON TE002.FDI_0001 = TS011.FDI_T004
                LEFT JOIN TS042 ON TE002.FDI_0001 = TS042.FDI_T096
                LEFT JOIN TS043 ON TE002.FDI_0001 = TS043.FDI_T047
                LEFT JOIN TS054 ON TE002.FDI_0001 = TS054.FDI_T508
                LEFT JOIN TS077 ON TE002.FDI_0001 = TS077.FDI_T618
                LIMIT $batchSize OFFSET $offset";

    $pdo->exec($joinSQL);
    logMessage("Batch inserito con offset $offset.");

    $offset += $batchSize;
}

$end = microtime(true);
$execution_time = ($end - $start);
logMessage("Join dei dati completato nel database per i primi 100 record in $execution_time secondi.");

// Verifica il contenuto della tabella joined_data
$stmt = $pdo->query("SELECT COUNT(*) AS count FROM $joinedTable");
$row = $stmt->fetch(PDO::FETCH_ASSOC);
logMessage("Numero di righe nella tabella joined_data: " . $row['count']);

if ($row['count'] == 0) {
    logMessage("Nessun dato trovato nella tabella joined_data.");
    exit;
}

// Controlla un campione di record per il debug
$sampleSize = 10; // Numero di record da controllare
$stmt = $pdo->query("SELECT * FROM $joinedTable LIMIT $sampleSize");

while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
    logMessage("Esempio record - Chiave: " . $row['keyValue']);
    foreach (array_keys($tables) as $table) {
        foreach ($row as $column => $value) {
            if (strpos($column, $table . "_") === 0) {
                if ($value) {
                    logMessage("Contenuto per $column: " . htmlspecialchars($value));
                } else {
                    logMessage("Campi vuoti per il prodotto con chiave " . $row['keyValue'] . " nella colonna " . $column);
                }
            }
        }
    }
}

// Esporta i risultati uniti in un file XML
$writer = new XMLWriter();
$writer->openURI($output_file);
$writer->startDocument('1.0', 'UTF-8');
$writer->setIndent(true);

$writer->startElement('Products');

$stmt = $pdo->query("SELECT * FROM $joinedTable");

while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
    $writer->startElement('Product');
    $writer->writeAttribute('ID', $row['keyValue']);
    foreach (array_keys($tables) as $table) {
        foreach ($row as $column => $value) {
            if (strpos($column, $table . "_") === 0 && $value !== null) {
                $writer->writeElement(substr($column, strlen($table) + 1), $value);
            }
        }
    }
    $writer->endElement(); // End Product element
}

$writer->endElement(); // End Products element
$writer->endDocument();
$writer->flush();

logMessage("Dati uniti esportati in '$output_file'.");
?>
