<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
ini_set('memory_limit', '8192M'); 
error_reporting(E_ALL);

$output_dir = __DIR__ . '/scaricati/';
$csv_dir = __DIR__ . '/scaricati/';

// Delete all existing CSV files in the directory
foreach (glob("$csv_dir/*.csv") as $csvFile) {
    unlink($csvFile);
    echo "\033[31mDeleted existing CSV file:\033[0m $csvFile\n";
}

if (!file_exists($csv_dir)) {
    mkdir($csv_dir, 0777, true);
}

$batchSize = 1000;
$flushInterval = 10; 
$logInterval = 10000; 

$zipFiles = glob("$output_dir/*.zip");
$totalZipFiles = count($zipFiles);
$totalStartTime = microtime(true); 
$totalRowsProcessed = 0; 
$totalExtractionTime = 0;

foreach ($zipFiles as $index => $zipFile) {
    echo "\033[34mStarting to process file:\033[0m $zipFile\n";

    if (!is_readable($zipFile)) {
        echo "\033[31mError: Cannot read file:\033[0m $zipFile\n";
        continue;
    }

    $fileSize = filesize($zipFile);
    echo "\033[34mCompressed file size:\033[0m " . ($fileSize / 1024 / 1024) . " MB\n";

    echo "\033[36mFile " . ($index + 1) . " of $totalZipFiles ZIPs\033[0m\n"; 

    $zip = new ZipArchive();
    echo "\033[32mOpening ZIP archive...\033[0m\n"; 
    $extractionStartTime = microtime(true);
    if ($zip->open($zipFile) !== TRUE) {
        echo "\033[31mError: Cannot open zip file:\033[0m $zipFile\n";
        continue;
    }

    for ($i = 0; $i < $zip->numFiles; $i++) {
        $xmlFilename = $zip->getNameIndex($i);
        if (pathinfo($xmlFilename, PATHINFO_EXTENSION) !== 'xml') {
            continue; // Skip non-XML files
        }

        $csvFile = $csv_dir . basename($xmlFilename, '.xml') . '.csv';
        $file = fopen($csvFile, 'w');
        $headers = [];
        $rowCount = 0;
        $startTime = microtime(true);

        echo "\033[32mReading XML file:\033[0m $xmlFilename\n"; 
        $xmlString = $zip->getFromIndex($i);
        $xml = new DOMDocument();
        $xml->loadXML($xmlString);

        $xpath = new DOMXPath($xml);
        $records = $xpath->query('//RECORD'); 
        $totalRecords = $records->length;

        if ($totalRecords == 0) {
            echo "\033[31mError: No RECORD elements found in the XML file:\033[0m $xmlFilename.\n";
            continue; 
        }

        // Extract headers from the first record
        $firstRecord = $records->item(0);
        foreach ($firstRecord->childNodes as $child) {
            if ($child->nodeType === XML_ELEMENT_NODE) {
                $headers[] = $child->nodeName;
            }
        }
        fputcsv($file, $headers); // Use fputcsv for proper CSV formatting
        echo "\033[35mHeaders:\033[0m " . implode(', ', $headers) . "\n";

        foreach ($records as $record) {
            $row = [];
            foreach ($headers as $header) {
                $elements = $xpath->query($header, $record); 
                $row[] = $elements->length > 0 ? $elements->item(0)->nodeValue : '';
            }

            if (!empty(array_filter($row))) {
                fputcsv($file, $row); // Use fputcsv for proper CSV formatting
                $rowCount++;
                $totalRowsProcessed++; 

                if ($rowCount % $flushInterval == 0) { 
                    fflush($file);
                }

                if ($rowCount % $logInterval == 0 || $rowCount == $totalRecords) { 
                    $endTime = microtime(true);
                    $elapsedTime = round($endTime - $startTime); 

                    $memoryUsage = memory_get_usage(true);
                    $memoryPeak = memory_get_peak_usage(true);
                    $memoryLimit = ini_get('memory_limit');
                    $memoryUsageFormatted = formatBytes($memoryUsage);
                    $memoryPeakFormatted = formatBytes($memoryPeak);

                    $percentComplete = round(($rowCount / $totalRecords) * 100, 2);

                    echo "\033[33mProcessed $rowCount rows of $totalRecords ($percentComplete%) in $elapsedTime seconds (Memory: $memoryUsageFormatted / $memoryPeakFormatted, Limit: $memoryLimit)\033[0m\n"; 
                    $startTime = microtime(true);

                    gc_collect_cycles();
                }
            }
        }
        
        fclose($file);
        echo "\033[32mConverted $xmlFilename to $csvFile. Rows written: $rowCount\033[0m\n";
    }

    $zip->close(); 

    $extractionEndTime = microtime(true); 
    $elapsedExtractionTime = round($extractionEndTime - $extractionStartTime);
    $totalExtractionTime += $elapsedExtractionTime; 
    echo "\033[32mZIP extraction time: $elapsedExtractionTime seconds\033[0m\n";
}

$totalEndTime = microtime(true);
$totalElapsedTime = round($totalEndTime - $totalStartTime);
echo "\033[31mTotal processing time: $totalElapsedTime seconds (Extraction: $totalExtractionTime seconds)\033[0m\n"; 

function formatBytes($bytes) {
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    $i = 0;
    while ($bytes >= 1024) {
        $bytes /= 1024;
        $i++;
    }
    return round($bytes, 2) . ' ' . $units[$i];
}